
;====================
; AVROT project
; Antenna rotor controller
; (C) Pavel Vachal,OK1DX 2002
; version 1.1
; 
; HAMWARE - free for Amateur radio community
; commercial use only with author's approval
;====================
; history: initial release 1.0
; 1.1 :M2 protocol adjusted to Nova software: data msg change,
;      longer TX buffer, msg sent also by A and E commands

.NOLIST

;*********  FUNCTION LIST  *********
; LOOP	- main program loop, started after initialisation
; SNDUPD - sends update to PC (M2 protokol)
; WAIT - timing loop
; ELCONV - converts EL ADC value to degrees
; AZCONV - converts AZ ADC value to degrees
; D2STR - converts degrees to ASCII string (with 360 deg rounding)
; N2STR - converts degrees to ASCII string (as it is)
; PRNR - prints ASCII string on LCD
; TEXT - sends a standart text stored in EEPROM to LCD
; LCDC - sends a command byte to LCD
; LCDD - sends a data byte to LCD
; PARNR - decodes ASCII number received from PC
; AZC - calculation of optimum AZ setpoint
; EEWR - store config data into EEPROM
; EERD - restore config data from EEPROM

;******** INTERRUPT HANDLERS *******
; RESET - entry point
; TIM0_OVF - timer 0 overflow: starts ADC, timeout motor counters, keyboard timing,
;			-5V generator, M2 autoupdate
; UART_RXC - RX complete; stores data from COM port into buffer, sends signal on end of cmd
; UART_DRE - TX empty; sends next byte to COM port, sends signal when all sent
; ADCQRV - AD converter data available; reads AZ and EL voltage, sends signal when data complete

.include	"4433def.inc"
.LIST

.DSEG	;SRAM registers

;EEPROM storage area start (configuration data restored from EEPROM)
AZSW1:	.BYTE 2	; AZ end switch CCW degrees
AZMAX:	.BYTE 2	; AZ rotation range
AZAD1:	.BYTE 2	; AZ ADC value for CCW switch
AZAD2:	.BYTE 2	; AZ ADC value for CW switch
ELSW1:	.BYTE 1	; EL minimum elevation
ELMAX:	.BYTE 1	; EL rotation range
ELAD1:	.BYTE 2	; EL ADC value for min elevation
ELAD2:	.BYTE 2	; EL ADC value for max elevation
DEAD:	.BYTE 1	; dead zone of rotor (if command value is very close to current one, no move)
;EEPROM storage area end (15 bytes)

AZADC:	.BYTE 2	; current AZ ADC value
AZDEG:	.BYTE 2	; current AZ in degrees (can be more than 360)
AZCMD:	.BYTE 2 ; command in deg (0-359)
AZSP:	.BYTE 2	; setpoint in deg (0 - ...)
ELADC:	.BYTE 2	; current EL ADC value
ELDEG:	.BYTE 1	; current EL in degrees
ELCMD:	.BYTE 1	; command in deg (0-180)
ELSP:	.BYTE 1	; setpoint in deg
ADTMP:	.BYTE 2	; temporary storage of AD convertor data (for averaging)
XTIM0:	.BYTE 1	; incremented every time TIMER0 overflows (scheduling of tasks)
AZCTR:	.BYTE 1	; AZ timeout counter
ELCTR:	.BYTE 1	; EL timeout counter
M2STS:	.BYTE 1	; M2 protocol status
				; b0-b3 speed, b4=1 evelation, b4=0 azimuth mode; b5 continuos update


TXBUF:	.BYTE 20 ;UART TX buffer - data
TXBP:	.BYTE 1	; current pointer in buffer (char that will be sent)
TXBE:	.BYTE 1	; pointer to end of data (1 position above last TX char)
.EQU	TXBL	= TXBP-TXBUF	;TX buffer length

RXBUF:	.BYTE 16 ;UART RX buffer - data
RXBP:	.BYTE 1	; current pointer (1 position above last rcvd char)
.EQU	RXBL	= RXBP-RXBUF	; RX buffer length
KEYCTR:	.BYTE 1 ; keyboard timer
KEYID:	.BYTE 1	; b0,b1 switch number 0=NIL, 1=S1, 2=S2, 3=S3; b2 = after 100 msec;
				; b3 = after 2 sec and than after 200 msec
				; b2 and b3 must be cleared by exe program
				
_AZSW1:	.BYTE 2	; temporary storage of cnfg parameters during setup
_AZMAX:	.BYTE 2
_AZAD1:	.BYTE 2
_AZAD2:	.BYTE 2
_ELSW1:	.BYTE 1
_ELMAX:	.BYTE 1
_ELAD1:	.BYTE 2
_ELAD2:	.BYTE 2
_DEAD:	.BYTE 1

AZT1:	.BYTE 1	; Timer 1 timeout registers - checking maximum allowed time of one movement (2 minutes)
ELT1:	.BYTE 1

; port initialisation
; DDR 1=output, 0=input
; PORT when input 1=pullup 0=tri state
; Xtal freq = 4 MHz
.EQU	_DDRB	=7		; PB0-PB2 out, PB3-PB5 inp pullup
.EQU	_PORTB	=0x38
.EQU	_DDRC	=0x3C	; PC0,1 inp ADC, PC2-PC5 out
.EQU	_PORTC	=0
.EQU	_DDRD	=0xF2	; PD0 inp, PD1 out, PD2,3 inp, PD4-PD7 out
.EQU	_PORTD	=0
.EQU	_ADCSR	=0x8E	; ADC enabled,prescaler 1:64 (ADC clock 62.5 kHz), INT enabled
.EQU	_UCSRB	=0x98	; RX,TX enable, RX complete INT enable
.EQU	_UBRRH	=0		; high byte of UART baud constant
.EQU	_UBRR	=25		; low byte - 9600 Bd by 4 MHz clock
.EQU	CHKSEED	=0xAA	; EEPROM checksum seed
.EQU	ESTART	=AZSW1	; starting addr in SRAM for EEPROM stored data
.EQU	EELEN	=AZADC-AZSW1	;length of data being stored in EEPROM
.EQU	LCDWT	=200	; timing const 150 mikrosec
.EQU	LCDW0	=250	; time delay 25 msec
.EQU	LCDW1	=150	; time delay 15 msec
.EQU	LCDW2	=40		; time delay 4 msec
.EQU	LCDAZ	=0x83		; current Azimuth line 1 posn 3
.EQU	LCDAZC	=0x88		; AZ command	line 1 posn 8
.EQU	LCDEL	=0xC3		; current Elevation line 2 posn 3
.EQU	LCDELC	=0xC8		; EL command	line 2 posn 8
.EQU	LCDADC	=0x8D		; ADC value during setup	line 1 posn 13
.EQU	LCDSET	=0xCB		; SETUP position (keyboard controlled) line 2 pos 12
.EQU	CMDCHAR	=0xD		; <CR> - terminates RX tlg command
.EQU	CMDCHAR2	=0xA	; <LF> - it is ignored
.EQU	_M2STS	=8			; speed=9, azimuth, autoupdate off
.EQU	RTOUT	=100		; pause when reversing motor direction x10 msec (1sec)
.EQU	T1L		=0x7C		; Timer 1 low/high byte ini value for 2 sec delay (prescaller 1024)
.EQU	T1H		=0xE1
.EQU	T1OUT	=60			; motion timeout (when rotor failed to reach pos in time limit) 60x2=120 sec

;LCD connection:
; PD4 = D0, PD5 = D1, PD6 = D2, PD7 = D3, PB1 = E, PB2 = R/S

; registers

.DEF	ISREG	=R15	; temp SREG storage during processing of INT handler
.DEF	TMP1	=R16	; general temp regs used in main loop
.DEF	TMP2	=R17
.DEF	TMP3	=R18
.DEF	ITMP1	=R19	; general temp regs used in INT handlers
.DEF	ITMP2	=R20
.DEF	ITMP3	=R21
.DEF	ATTR	=R25	; temp used in optimal rotation calc (bit7 - CW, 6 - D<=180, 5 - in range)
.DEF	COMST	=R26	; serial communication status
						; bit 0 - TX in progress (buffer busy)
						; bit 1 - RX tlg ready to be evaluated
						; bit 6 - send update flag
						; bit 7 - setup active
; move direction 1=degrees increasing (CW, UP), 0=degrees decreasing (CCW,DWN)
.DEF	RFLAG	=R27	; bit 0 - 1: even ADC; calculation will be started
						; bit 1 - AZ last move direction
						; bit 2 - AZ new move direction (result of calc)
						; bit 3 - EL last move direction
						; bit 4 - EL new move direction
						; bit 5 - 1=new ADC data available (set by INT rutine, cleared by display rutine)
						; bit 6 - AZ request to move
						; bit 7 - EL request to move
						
; Y register (r28 LOW BYTE, r29 HIGH BYTE - always zero) used in INT handlers
; Z register (r30 LOW BYTE, r31 HIGH BYTE - always zero) used in main loop

.CSEG
.ORG	0

	
	rjmp	RESET		;	Reset Handler
	reti 	; IRQ0 Handler
	reti	; IRQ1 Handler
	reti	;rjmp TIM1_CAPT ; Timer1 Capture Handler
	reti	;rjmp TIM1_COMP ; Timer1 compare Handler
	rjmp 	TIM1_OVF ; Timer1 Overflow Handler
	rjmp	TIM0_OVF ; Timer0 Overflow Handler
	reti	; SPI Transfer Complete Handler
	rjmp 	UART_RXC ; UART RX Complete Handler
	rjmp 	UART_DRE ; UDR Empty Handler
	reti	;rjmp UART_TXC ; UART TX Complete Handler
	rjmp	ADCQRV	;rjmp ADC ; ADC Conversion Complete Interrupt Handler
	reti	; EEPROM Ready Handler
	reti	; Analog Comparator Handler
	
RESET:
	ldi	TMP1,RAMEND
	out	SP,TMP1		; stack

	ldi	TMP1,_DDRB	; port mode ini
	out	DDRB,TMP1
	ldi	TMP1,_DDRC
	out	DDRC,TMP1
	ldi	TMP1,_DDRD
	out	DDRD,TMP1
	ldi	TMP1,_PORTB
	out	PORTB,TMP1
	ldi	TMP1,_PORTC
	out	PORTC,TMP1
	ldi	TMP1,_PORTD
	out	PORTD,TMP1
;		
; LCD MODULE INITIALIZATION
;
	ldi	TMP1,250		; wait 25 msec
	rcall WAIT
;
	ldi	TMP1,0x30
	rcall LCDX			; 8 bit mode (+ blabla)
	ldi	TMP1,40			; wait 4 msec
	rcall WAIT
;
	ldi	TMP1,0x30
	rcall LCDX			; 8 bit mode (+ blabla)
	ldi	TMP1,40			; wait 4 msec
	rcall WAIT
;
	ldi	TMP1,0x30
	rcall LCDX			; 8 bit mode (+ blabla)
	rcall MOM
;
	ldi	TMP1,0x20
	rcall LCDX			; 4 bit mode (+ blabla)
	rcall MOM
;
	ldi	TMP1,0x28
	rcall LCDC			; 4 bits, 2 lines, 5x7 dots
;
	ldi	TMP1,0x08
	rcall LCDC			; display OFF
;
	ldi TMP1,0x0C
	rcall LCDC			; display ON, no cursor, no blink
;
	ldi TMP1,0x06
	rcall LCDC			; cursor move increment, no display shift
;
	ldi	TMP1,0x01	
	rcall LCDC			; clear display
;
	ldi	TMP1,40			; wait 4 msec
	rcall WAIT

	ldi	TMP1,VERS
	rcall	TEXT		; AVROT version OK1DX
	ldi	TMP1,VERT
	rcall	TEXT		; KE6LVK
;lp:	rjmp lp

	ldi	TMP3,100		; 100ms
VER1:	ldi TMP1,150	; !!!! change to 150 for release version
	rcall	WAIT
	dec	TMP3
	brne	VER1		; cca 1 sec
	rcall	INITXT		; initial display text
;	
; M2 protocol ini
;
	ldi	TMP1,_M2STS
	sts	M2STS,TMP1
;
; read EEPROM rotor configuration data
;
	rcall	EERD
;	
; keyboard timers
;
	clr	TMP1
	sts	KEYID,TMP1
	sts	KEYCTR,TMP1
;	
; A/D convertor initialization
;
	clr	RFLAG
	out	ADMUX,TMP1		; channel 0 selected
	ldi	TMP1,_ADCSR
	out	ADCSR,TMP1		; ADC mode
;	
; UART initialisation
;
	ldi	TMP1,RXBUF
	sts	RXBP,TMP1		; ini of RX buffer pointer
	ldi	TMP1,TXBUF
	sts	TXBP,TMP1		; ini of TX buffer pointers
	sts	TXBE,TMP1
	ldi	TMP1,_UBRRH
	out	UBRRH,TMP1
	ldi	TMP1,_UBRR
	out	UBRR,TMP1		;speed const
	ldi	TMP1,_UCSRB
	out	UCSRB,TMP1		; mode: 8bit, RX enable, TX enable, RX INT enable
						; UCSRA left on default
	clr	COMST
;
; TIMER 0 + 1 initialisation
;
	ldi	TMP1,5
	out	TCCR0,TMP1		; using internal clock / 1024
	out	TCCR1B,TMP1

	clr	TMP1
	out	TCCR1A,TMP1

	ldi	ITMP1,T1H		;load new timer value

	ldi	TMP1,T1H
	out	TCNT1H,TMP1		;initial timer 1 value

	ldi	TMP1,T1L
	out	TCNT1L,TMP1

	ldi	TMP1,0x82
	out	TIMSK,TMP1		; enable both TIMER0 and TIMER1 overflow interrupt
	
	sei					;enable global interrupt; the fun start NOW!
;
; ********** MAIN PROGRAM LOOP *********
; it waits for events signalled by interrupt rutines
;	
LOOP:
		;**** based on ADC data: STOP THE MOTOR WHEN SETPOINT REACHED, DISPLAY CURRENT AZ/EL
		sbrs	RFLAG,5		;ADC data ready
		rjmp	LOOP1
		cbr		RFLAG,1<<5	;clear the flag
		rcall	AZCONV			;AZ conversion ADC -> deg
		rcall	ELCONV			;EL conversion ADC -> deg
	;AZIMUTH		
		sbis	PORTC,2	; moving CW?
		rjmp	AZY1	; no...
		lds	TMP1,AZDEG+1
		lds	TMP2,AZSP+1
		sub	TMP1,TMP2
		brne	AZY2
		lds	TMP1,AZDEG	; when MSB bytes the same
		lds	TMP2,AZSP
		sub	TMP1,TMP2
AZY2:	brcc	AZYS	; CY clear -> [AZDEG]>=[AZSP]
		rjmp	AZY4
AZY1:	sbis	PORTC,3	; moving CCW?
		rjmp	ELY		; no, AZ stopped... process elevation and display data
		lds	TMP1,AZSP+1
		lds	TMP2,AZDEG+1
		sub	TMP1,TMP2
		brne	AZY3
		lds	TMP1,AZSP
		lds	TMP2,AZDEG
		sub	TMP1,TMP2
AZY3:	brcs	AZY4		; CY clear -> [AZSP]>=[AZDEG]
AZYS:	cbi	PORTC,2		; STOP AZ MOTOR!!!!
		cbi	PORTC,3
		ldi	TMP1,T1STP
		rcall	TEXT	; LCD msg
		ldi	TMP1,RTOUT	; start reverz timeout
		sts	AZCTR,TMP1
		rjmp	ELY
AZY4:	lds	TMP1,AZT1	;stop the motor when timeout counter == 0
		and	TMP1,TMP1
		breq	AZYS	
		
	;ELEVATION		
ELY:	sbis	PORTC,5	; moving UP?
		rjmp	ELY1	; no...
		lds	TMP1,ELDEG	; when MSB bytes the same
		lds	TMP2,ELSP
		sub	TMP1,TMP2
		brcc	ELYS	; CY clear -> [ELDEG]>=[ELSP]
		rjmp	ELY2
ELY1:	sbis	PORTC,4	; moving DWN?
		rjmp	DDD		; no, AZ stopped; display only
		lds	TMP1,ELSP
		lds	TMP2,ELDEG
		sub	TMP1,TMP2
		brcs	ELY2		; CY clear -> [ELSP]>=[ELDEG]
ELYS:	cbi	PORTC,4		; STOP EL MOTOR!!!!
		cbi	PORTC,5
		ldi	TMP1,T2STP
		rcall	TEXT	; LCD msg
		ldi	TMP1,RTOUT	; start reverz timeout
		sts	ELCTR,TMP1
		rjmp	DDD
ELY2:	lds	TMP1,ELT1	;stop motor when motion timeout == 0
		and	TMP1,TMP1
		breq	ELYS
		
DDD:		; display current AZ and EL
		lds	r8,AZDEG
		lds	r9,AZDEG+1
		rcall	D2ASCI
		ldi	TMP1,LCDAZ
		rcall	PRNR
		
		lds	r8,ELDEG
		clr	r9
		rcall	N2ASCI
		ldi	TMP1,LCDEL
		rcall	PRNR

LOOP1:	
		sbrs	COMST,1
		rjmp	LOOP3

; **** COMMAND RECEIVED FROM UART
		lds	TMP1,RXBP			;REQUEST FOR MANUAL UPDATE
		ldi	TMP2,RXBUF
		cp	TMP1,TMP2			;RX buffer empty
		brne	CMD
		cbr	COMST,2
		rjmp	LOOPS			;send update only
		
CMD:	lds	TMP1,RXBUF
		cpi TMP1,'a'			;conversion to uppercase
		brcs	CMD0
		subi	TMP1,0x20
		
CMD0:	cpi	TMP1,'A'			;SET AZIMUTH MODE + send update
		brne CMD1
		lds	TMP1, M2STS
		cbr	TMP1,0x10		;bit 4
		sts	M2STS,TMP1
		rjmp	LOOPS
	
CMD1:	cpi TMP1,'E'			;SET ELEVATION MODE + send update
		brne CMD3
		lds	TMP1,M2STS
		sbr	TMP1,0x10		;bit 4
		sts	M2STS,TMP1
		rjmp	LOOPS
	
CMD3:	cpi	TMP1,'U'			;START AUTOUPDATE MODE
		brne	CMD4
		lds	TMP1, M2STS
		sbr	TMP1,0x20		;bit 5
		sts	M2STS,TMP1
		rjmp	LOOP2
		
CMD4:	cpi	TMP1,'N'			;STOP AUTOUPDATE MODE
		brne	CMD5
		lds	TMP1, M2STS
		cbr	TMP1,0x20		;bit 5
		sts	M2STS,TMP1
		rjmp	LOOP2
		
CMD5:	cpi	TMP1,'0'			;NUMBER - NEW SETPOINT VALUE
		brcs	CMD6
		cpi	TMP1,'9'+1
		brcc	CMD6
		ldi	r30,RXBUF
		rcall	PARNR		;conversion to number
		lds	TMP2,M2STS
		sbrc	TMP2,4		;AZ or EL
		rjmp	CMD5A
		ldi	TMP1,1
		cp	TMP1,r5
		brne	CMD5C
		ldi	TMP1,0x68			; 360 = 0x168
		cp	TMP1,r4
CMD5C:	brcs	CMD5D		; AZ cmd >360 - ERROR 3		
		sts	AZCMD,r4
		sts	AZCMD+1,r5
		rcall	AZC			; calculates best SETPOINT and direction
		sbr	RFLAG,1<<6		;set AZ request flag
		ldi	TMP1,LCDAZC
		rjmp	CMD5B	
CMD5A:	and	r5,r5			; EL upper cmd byte must be 0
		brne	CMD5D
		ldi	TMP1,240
		cp	TMP1,r4
		brcs	CMD5D		; EL cmd max. 240 degs (60+180)
		sts	ELCMD,r4
		rcall	ELC			; calculates setpoint / dir
		sbr RFLAG,1<<7		;set EL request flag
		ldi	TMP1,LCDELC
CMD5B:	ldi	TMP2,RXBUF
		sts	RXBP,TMP2	;buffer pointer back to begin
		cbr	COMST,2		;allow further reception
		rcall	PRNR	;display command data
		rjmp	LOOP2
CMD5D:	rjmp	ERR3	;alias for ERR3
		
CMD6:	cpi	TMP1,'W'	;GS232 command / example "W223 56<cr><lf>"
		brne	CMD2
		ldi	r30,RXBUF
		inc	r30
		rcall	PARNR	;decode AZ value
		sts	AZCMD,r4
		sts	AZCMD+1,r5
		mov	r7,r1
		mov	r8,r2
		mov	r9,r3		;ASCII string temp storage
		inc	r30			;jump over one space char
		rcall	PARNR	;decode EL value
		sts	ELCMD,r4
		ldi	TMP2,RXBUF
		sts	RXBP,TMP2	;RX buffer initialisation
		cbr	COMST,2		;allow further reception; slow print/calc will not block it
		ldi	TMP1,LCDELC
		rcall	PRNR	;display EL command data
		ldi	TMP1,LCDAZC
		mov	r1,r7
		mov	r2,r8
		mov	r3,r9
		rcall	PRNR	;display AZ command data
		rcall	ELC		;EL setpoint/direction calculation
		sbr	RFLAG,1<<7
		rcall	AZC		;AZ SETPOINT/direction calculation
		sbr	RFLAG,1<<6
		rjmp	LOOP2
		
CMD2:	cpi TMP1,'S'			;SPEED OR STOP THE ROTOR
		brne	CMD7
		lds	TMP1,RXBP
		cpi	TMP1,RXBUF+1		;just 1 character in buffer->stop command
		breq	CMD2A
		lds	TMP1,RXBUF+1		;following character
		subi	TMP1,'1'		;allowed numbers 1-9
		brcs	ERR3
		cpi	TMP1,9	
		brcc	ERR3
		lds	TMP2,M2STS
		cbr	TMP2,0x0F			;clear all speed bits
		add	TMP2,TMP1			;add new speed (0 - 8)
		sts	M2STS,TMP2
		rjmp	LOOP2
CMD2A:	lds	TMP1,M2STS
		sbrc	TMP1,4		;AZ or EL???
		rjmp	CMD2B
		cbi	PORTC,2
		cbi	PORTC,3			; STOP AZ motor
		ldi	TMP1,T1STP
		rjmp	CMD2C
CMD2B:	cbi	PORTC,4
		cbi	PORTC,5			; STOP EL motor
		ldi	TMP1,T2STP
CMD2C:	rcall	TEXT		;LCD msg
		rjmp	LOOP2
		
CMD7:		
	
ERR3:	ldi	TMP2,0x03	;error number
		rcall	SNDERR

LOOPS:	rcall	SNDUPD
LOOP2:	ldi	TMP1,RXBUF
		sts	RXBP,TMP1	;buffer pointer back to begin
		cbr	COMST,2		;clear 'RX data ready' attribute (new tlg can arrive)
LOOP3:	sbrs	COMST,6	; *** SEND AUTOUPDATE update attribute
		rjmp	LOOP4
		cbr		COMST,0x40	;clear the attr
		rcall	SNDUPD
		
LOOP4:; **** AZIMUTH MOTOR CONTROL - response on command
		sbrs	RFLAG,6	; AZ REQ - new command /AZ data received??? 
		rjmp	LOOP5
		sbis	PORTC,2	; AZ moving CW??
		rjmp	AZX1
		sbrs	RFLAG,2	; AZ command also CW?
		rjmp	AZX2
		rjmp	AZXC	; yes, moving in same direction; no change needed
AZX1:	sbis	PORTC,3	; AZ moving CCW?
		rjmp	AZX3
		sbrs	RFLAG,2	; AZ command also CCW?
		rjmp	AZXC	; yes, same direction, no change
AZX2:	cbi	PORTC,2		; AZ CW motor STOP
		cbi	PORTC,3		; AZ CCW motor STOP
		ldi	TMP1,T1STP
		rcall	TEXT	; LCD msg
		ldi	TMP1, RTOUT	; delay when reversing motor 100 x 10 msec = 1 sec
		sts	AZCTR, TMP1
		rjmp	LOOP5	; leave REQ flag set
AZX3:	mov	TMP1,RFLAG
		lsl	TMP1
		eor	TMP1,RFLAG
		sbrs	TMP1,2	; bit1 == bit2 ??
		rjmp	AZX4	; same direction, ignore reversing counter
		lds	TMP1,AZCTR	; different direction, danger of too fast motor reversal
		and	TMP1,TMP1
		brne	LOOP5	; timeout still running, no action
AZX4:	lds	TMP1,AZSP
		lds	TMP2,AZDEG
		sub	TMP1,TMP2
		lds	TMP2,AZSP+1
		lds	TMP3,AZDEG+1
		sbc	TMP2,TMP3	; TMP1/TMP2 now [AZSP]-[AZDEG]
		brpl	AZX5
		neg	TMP1
		brcs	AZX6
		dec	TMP2
AZX6:	com	TMP2		; TMP1/TMP2 = - TMP1/TMP2 (absolute difference between current and setpoint)
AZX5:	and	TMP2,TMP2
		brne	AZX7	;when TMP2 non zero, always out of deadzone
		lds	TMP2,DEAD	; dead zone width
		cp	TMP2,TMP1
		brcc	AZXC	;command inside dead zone; command ignored
AZX7:	cbi	PORTC,2
		cbi	PORTC,3		; for sure....
		sbrs RFLAG,2	; new AZ direction
		rjmp	AZX8
		sbi	PORTC,2		; CW rotation START <<<<
		sbr	RFLAG,1<<1	; last direction flag set
		ldi	TMP1,T_CW
		rjmp	AZX9
AZX8:	sbi PORTC,3		; CCW rotation START <<<<
		cbr	RFLAG,1<<1	; last direction flag cleared
		ldi	TMP1,T_CCW
AZX9:	ldi	TMP2,T1OUT
		sts	AZT1,TMP2
		rcall	TEXT	; LCD text
AZXC:	cbr	RFLAG,1<<6	; clear AZ REQ flag
	
	
LOOP5:
	; **** ELEVATION MOTOR CONTROL - response to command
		sbrs	RFLAG,7	; EL REQ - new command /EL data received??? 
		rjmp	LOOP6
		sbis	PORTC,5	; EL moving UP??
		rjmp	ELX1
		sbrs	RFLAG,4	; EL command also UP?
		rjmp	ELX2
		rjmp	ELXC	; yes, moving in same direction; no change needed
ELX1:	sbis	PORTC,4	; EL moving DWN?
		rjmp	ELX3	; no, EL motor stopped
		sbrs	RFLAG,4	; EL command also DWN?
		rjmp	ELXC	; yes, same direction, no change
ELX2:	cbi	PORTC,4		; EL DWN motor STOP
		cbi	PORTC,5		; EL UP motor STOP
		ldi	TMP1,T2STP
		rcall	TEXT	; LCD msg
		ldi	TMP1, RTOUT	; delay when reversing motor 100 x 10 msec = 1 sec
		sts	ELCTR, TMP1
		rjmp	LOOP6	; leave REQ flag set and continue...
ELX3:	mov	TMP1,RFLAG
		lsl	TMP1
		eor	TMP1,RFLAG
		sbrs	TMP1,4	; bit3 == bit4 ??
		rjmp	ELX4	; same direction, ignore reversing counter
		lds	TMP1,ELCTR	; different direction, danger of too fast motor reversal
		and	TMP1,TMP1
		brne	LOOP6	; timeout still running, no action, REQ flag unchanged
ELX4:	lds	TMP1,ELSP
		lds	TMP2,ELDEG
		sub	TMP1,TMP2	; TMP1 now [ELSP]-[ELDEG]
		brpl	ELX5
		neg	TMP1		; TMP1= - TMP1 (absolute difference between current and setpoint)
ELX5:	lds	TMP2,DEAD	; dead zone width
		cp	TMP2,TMP1
		brcc	ELXC	;command inside dead zone; command ignored
ELX7:	cbi	PORTC,4
		cbi	PORTC,5		; for sure....
		sbrs RFLAG,4	; new EL direction
		rjmp	ELX8
		sbi	PORTC,5		; UP rotation START <<<<
		sbr	RFLAG,1<<3	; last direction flag set
		ldi	TMP1,T_UP
		rjmp	ELX9
ELX8:	sbi PORTC,4		; DWN rotation START <<<<
		cbr	RFLAG,1<<3	; last direction flag cleared
		ldi	TMP1,T_DWN
ELX9:	ldi	TMP2,T1OUT	; motion timeout
		sts	ELT1,TMP2
		rcall	TEXT	; LCD text
ELXC:	cbr	RFLAG,1<<7	; clear EL REQ flag
		
LOOP6:	cli				; SETUP procedure called when S1 key hold for more than 2 sec
		lds	TMP1,KEYID
		mov TMP2,TMP1
		andi TMP1,3
		sts	KEYID,TMP1
		sei
		andi	TMP2,0x0B
		cpi	TMP2,9
		brne	LOOP7
		rcall	SETUP
LOOP7:
		rjmp	LOOP	;back in main loop

;======================================== end of main loop

;******* SEND UPDATE TO COMPUTER VIA COM PORT
; tlg 19 bytes long - 17 chars + <CR><LF>
; example : "A P= 012   S=9 MV<cr><lf>"
SNDUPD:	sbrc	COMST,0
		rjmp	SNDUPD		;loop when last tlg not yet sent (bit cleared by UART_DRE rutine)
		lds	r30,TXBE
		clr	r31
		lds	TMP3,M2STS		;AZ or EL ?
		ldi	TMP1,'A'
		sbrc TMP3,4
		ldi	TMP1,'E'
		st	Z+,TMP1
		ldi	TMP1,' '
		st Z+,TMP1
		ldi	TMP1,'P'
		st Z+,TMP1
		ldi	TMP1,'='
		st Z+,TMP1
		ldi	TMP1,' '		; space or sign - AVROT supplies positive values only, so space is used
		st Z+,TMP1
		sbrc	TMP3,4		;AZ or EL ?
		rjmp	SUPD1
		lds	r8,AZDEG
		lds	r9,AZDEG+1
		rjmp	SUPD2
SUPD1:	lds	r8,ELDEG
		clr	r9
SUPD2:	rcall D2ASCI		; convert to ASCII string
		st	Z+,r1			; store current degree value
		st	Z+,r2
		st	Z+,r3
		ldi	TMP1,' '
		st 	Z+,TMP1
		st 	Z+,TMP1
		st 	Z+,TMP1			; 3 spaces necessary
		ldi	TMP1,'S'
		st 	Z+,TMP1
		ldi	TMP1,'='
		st 	Z+,TMP1
		mov	TMP1,TMP3
		cbr	TMP1,0xF0		;mask other than speed value
		ldi	TMP2,'1'
		add	TMP1,TMP2
		st	Z+,TMP1			;ASCII speed value
		ldi	TMP1,' '
		st	Z+,TMP1
		lds	TMP3,M2STS
		in	TMP1,PORTC
		sbrc	TMP3,4		; AZ or EL???
		rjmp	SUPD5
		andi	TMP1,0x0C	;mask other than AZ CW and AZ CCW
		rjmp	SUPD6
SUPD5:	andi	TMP1,0x30	;mask other than EL UP and EL DWN
SUPD6:	brne	SUPD3		; non zero -> moving	
		ldi	TMP1,'S'
		st	Z+,TMP1
		ldi TMP1,'T'
		st	Z+,TMP1
		rjmp	SUPD4
SUPD3:	ldi	TMP1,'M'
		st	Z+,TMP1
		ldi TMP1,'V'
		st	Z+,TMP1
SUPD4:	ldi	TMP1,0x0d		; <CR>
		st	Z+,TMP1
		ldi	TMP1,0x0a		; <LF>
		st	Z+,TMP1
		sts	TXBE,r30	; store addr of 1st byte after tlg
		sbr	COMST,1		; tlg is being sent
		sbi	UCSRB,UDRIE	; enable interrupt from empty UART TX buf - go!				
		ret		

;****** sends an error message to PC ***
; input: error number in TMP2 (2 hex digits i.e. 0x87 -> ERR=87)
SNDERR:	
		sbrc	COMST,0 ;loop till ready
		rjmp	SNDERR
		lds	r30,TXBE
		clr	r31
		lds	TMP3,M2STS		;AZ or EL ?
		ldi	TMP1,'A'
		sbrc TMP3,4
		ldi	TMP1,'E'
		st	Z+,TMP1
		ldi TMP1,' '
		st	Z+,TMP1
		ldi	TMP1,'E'
		st	Z+,TMP1
		ldi	TMP1,'R'
		st	Z+,TMP1
		st	Z+,TMP1
		ldi	TMP1,'='
		st	Z+,TMP1
		ldi	TMP3,'0'
		mov	TMP1,TMP2
		swap TMP1
		andi TMP1,0x0F
		add	TMP1,TMP3
		st	Z+,TMP1
		andi TMP2,0x0F
		add	TMP2,TMP3
		st	Z+,TMP2
		rcall	SUPD4
		
	
WAIT:	ldi	TMP2,133	;timing procedure, value in TMP1 (x 100 mikrosec)
WAIT1:	dec	TMP2
		brne	WAIT1
		dec	TMP1
		brne	WAIT
		ret
		
INITXT:	ldi	TMP1,TXT1	; prints initial LCD text
		rcall	TEXT
		ldi	TMP1,TXT2
		rjmp	TEXT
;
; sends a standard text from EEPROM to LCD
; input: address of text in EEPROM in TMP1
;
TEXT:	mov	TMP3,TMP1
		out	EEAR,TMP3
		sbi	EECR,EERE
TEXT1:	sbic	EECR,EERE
		rjmp	TEXT1
		in	TMP1,EEDR	; text position byte, sent as command
		rcall	LCDC
TEXT4:	inc	TMP3
		out	EEAR,TMP3
		sbi	EECR,EERE
TEXT2:	sbic	EECR,EERE
		rjmp	TEXT2
		in	TMP1,EEDR	;ASCII character code 
		and	TMP1,TMP1
		breq	TEXT3	;0 ... string termination
		rcall	LCDD
		rjmp	TEXT4
TEXT3:	ret
		

LCDC:	cbi	PORTB,2		; sends TMP1 to LCD as command (RS=0)
		rjmp	LCD1
LCDD:	sbi PORTB,2		; sends TMP1 to LCD as data (RS=1)
LCD1:	rcall	LCDS
		swap	TMP1
		rcall	LCDS
MOM:	ldi	TMP1,LCDWT	; cekani 150 mikrosekund
LCD2:	dec	TMP1
		brne	LCD2
		ret		
LCDX:	cbi	PORTB,2		; send 4 upper bits as 1 byte command (RS=0)
LCDS:	mov	TMP2,TMP1	;sends 4 upper bits to LCD
		andi	TMP2,0xF0
		out	PORTD,TMP2
		sbi	PORTB,1			; E = 1
		nop
		cbi	PORTB,1			; E = 0 - write
		ret
		
;****** INTERRUPT HANDLER FOR UART RX COMPLETE *******
; stores received char into buffer
; sets flag when CMDCHAR received
UART_RXC:
		in	ISREG,SREG
		in	ITMP3,UDR	;received char
		sbrc	COMST,1	;if set, old command is not yet processed; new rx data ignored
		rjmp	RX2
		cpi	ITMP3,CMDCHAR2
		breq	RX2		;it is ignored
		cpi	ITMP3,CMDCHAR
		breq	RX3		;end of tlg char received (not stored in buffer)
		lds	ITMP1,RXBP
		ldi	ITMP2,RXBP	;address of first byte after RX buffer
		cp	ITMP1,ITMP2
		brcs	RX1
		ldi	ITMP1,RXBUF ;buffer overrun - old contents vasted and start again
		sts	TXBUF,ITMP3 ;store on first posn in buffer
		rjmp	RX2
RX1:	clr	r29			;store the chat into buffer on given posn
		mov r28,ITMP1
		st	Y,ITMP3
		inc ITMP1		; RXBP++
		sts	RXBP,ITMP1
		rjmp	RX2
RX3:	sbr	COMST,2		;tlg ready - set bit 1 in COMST
RX2:	out	SREG,ISREG
		reti
		
;****** INTERRUPT HANDLER FOR UART TX EMPTY ********
; sends one character from TX buffer to UART; when buffer sent completely,
; it initialises buffer pointers and disables INT from empty TX buffer
UART_DRE:
		in ISREG,SREG
		lds	ITMP1,TXBP
		lds	ITMP2,TXBE
		cp	ITMP1,ITMP2
		brcs	SND1
		cbi	UCSRB,UDRIE	; TXBP>=TXBE... end of tlg; disable INT
		cbr	COMST,1		; TX finished status (clear bit 0) - buffer available for new data
		ldi	ITMP1,TXBUF
		sts	TXBP,ITMP1	; pointers set to buffer begin
		sts	TXBE,ITMP1
		rjmp	SND2
SND1:	clr	r29
		mov	r28,ITMP1	; Y = [TXBP]
		ld	ITMP2,Y
		out	UDR,ITMP2	; char sent to UART
		inc	ITMP1		; TXBP++
		sts	TXBP,ITMP1
SND2:	out SREG,ISREG
		reti
		
		
;****** INTERRUPT HANDLER OF TIMER 0 (8 bit) **********
;called every 10 msec (100 times per second)	
TIM0_OVF:
		in	ISREG,SREG
		lds	ITMP1,AZCTR
		and	ITMP1,ITMP1
		breq	CTR1
		dec	ITMP1			;decrement till zero - Azimuth counter
		sts	AZCTR,ITMP1
CTR1:	lds	ITMP1,ELCTR
		and	ITMP1,ITMP1
		breq	CTR2
		dec	ITMP1			;decrement till zero - Elevation counter
		sts	ELCTR,ITMP1
CTR2:
		lds	ITMP3,KEYID		;control switches/keyboard handling
		in	ITMP1,PINB
		andi	ITMP1,0x38	; bits 3,4,5
		clr	ITMP2
		cpi	ITMP1,0x38
		breq	NOKEY		;no key pressed
		inc	ITMP2			;=1
		cpi	ITMP1,0x30
		breq	KEY4
		inc	ITMP2			;=2
		cpi	ITMP1,0x28
		breq	KEY4
		inc	ITMP2			;=3
		cpi	ITMP1,0x18
		brne	NOKEY		;2 keys together - same as no key
KEY4:	mov	ITMP1,ITMP3		;KEYID value
		andi	ITMP1,0x03	;2 LSB bits
		cp	ITMP1,ITMP2
		brne	NEWKEY		;different key pressed
		lds	ITMP1,KEYCTR
		inc	ITMP1
		cpi	ITMP1,10		;after 100 nsec
		brne	KEY1
		sbr	ITMP3,4			;set bit2
KEY1:	cpi	ITMP1,200		;after 2 sec
		brcs	KEY3
		sbr	ITMP3,8			;set bit3
		ldi	ITMP1,180		; after 200-180=20 (200 msec) autorepeated
KEY3:	sts	KEYCTR,ITMP1
		rjmp	KEY2
NOKEY:	clr	ITMP2
NEWKEY:	clr	ITMP1
		sts	KEYCTR,ITMP1
		andi	ITMP3,0xF0	;clear 4 LSB
		add	ITMP3,ITMP2
KEY2:	sts	KEYID,ITMP3
		
TIM03:			
		lds	ITMP1,XTIM0		;increment of subcounter register
		inc ITMP1
		sts	XTIM0,ITMP1
		cbi	PORTB,0		
		sbrs	ITMP1,0		;for -5V generator output
		sbi	PORTB,0
			
		mov	ITMP2,ITMP1
		andi	ITMP2,0xF	;every 16th cycle (4 LSB bits)
		brne	TIM01
		cbi	ADMUX,0			;AD channel = 0 (AZ)
		sbi	ADCSR,ADSC		;start conversion (every 16x10=160 msec)
TIM01:	lds	ITMP2,M2STS
		sbrs	ITMP2,5		;autoupdate
		rjmp	TIM02
		mov	ITMP2,ITMP1
		andi	ITMP2,0x7F	;every 128th cycle (7 LSB bits) = 1.28 sec
		brne	TIM02
		sbr	COMST,0x40		;set bit 6 in M2STS - send update!	
TIM02:	ldi ITMP1,217		;new timer starting value
		out	TCNT0,ITMP1
		out	SREG,ISREG
		reti

; ****** TIMER 1 INTERRUPT HANDLER ****** performed every 2 seconds

TIM1_OVF:
		in	ISREG,SREG
		lds	ITMP1,AZT1
		and	ITMP1,ITMP1
		breq	T1R1
		dec	ITMP1			;decrement till zero - Azimuth motion timeout
		sts	AZT1,ITMP1
T1R1:	lds	ITMP1,ELT1
		and	ITMP1,ITMP1
		breq	T1R2
		dec	ITMP1			;decrement till zero - Elevation motion timeout
		sts	ELT1,ITMP1
T1R2:	ldi	ITMP1,T1H		;load new timer value
		out	TCNT1H,ITMP1
		ldi	ITMP1,T1L
		out	TCNT1L,ITMP1
		out	SREG,ISREG
		reti

;****** INTERRUPT HANDLER OF ANALOG / DIGITAL CONVERTER *******
; complete read cycle consist of 4 A/D conversions, 2 AZ and 2 EL.
; average value is calculated for each pair
; the cycle is started from timer 0 interrupt rutine
; by starting first conversion
; channel mutiplex is set to channel 0 (AZ) there
; when finished, both AZ and EL values are stored and bit 5 in RFLAG set
ADCQRV:					; called when AD conversion finished
		in	ISREG,SREG
		in	ITMP1,ADCL
		in	ITMP2,ADCH
		andi	ITMP2,3		; only 2 bits valid
		sbrc	RFLAG,0
		rjmp	ADC1
		sts	ADTMP,ITMP1		; no calculation, just temporary storage
		sts	ADTMP+1,ITMP2
		sbr	RFLAG,1			; next time it will calculate
ADC4:	sbi	ADCSR,ADSC		; start next convertion
		rjmp	ADC6
		
ADC1:	cbr	RFLAG,1			; next convertion will store data only
		lds	ITMP3,ADTMP
		add	ITMP1,ITMP3
		lds	ITMP3,ADTMP+1
		adc	ITMP2,ITMP3		; last value + new value
		lsr	ITMP2
		ror	ITMP1			; : 2 - average value
		sbic	ADMUX,0		;0=AZ, 1=EL
		rjmp ADC2
		sts	AZADC,ITMP1		; AZ calculation
		sts	AZADC+1,ITMP2
ADC3:	ldi	ITMP1,1			; select ADC channel 1 (elevation)
		out	ADMUX,ITMP1
		rjmp ADC4
ADC2:	sts	ELADC,ITMP1
		sts	ELADC+1,ITMP2
ADC5:	sbr	RFLAG,1<<5		;signal new data available (to main loop)
ADC6:	out	SREG,ISREG
		reti				; without start of conversion!!


;******* DEGREES TO TEXT STRING	
; input r8,r9 (can be more than 360 degs)
; output: r1,r2,r3 ASCII characters/string '000' to '359'
D2ASCI:
		mov	r5,r8			; (r5,r6) = (r8,r9) - 360 deg  {360 = 1*256 + 104}
		mov	r6,r9
A2D4:	ldi	TMP1,104
		sub	r5,TMP1
		ldi	TMP1,1
		sbc	r6,TMP1
		brcs	N2ASCI		; carry set - (r8,r9) is between 0 and 359 deg
		mov	r8,r5
		mov	r9,r6
		rjmp	A2D4

;******* NUMBER TO TEXT STRING	
; input r8,r9 (0 to 999; 1003 displayed as ':03'
; output: r1,r2,r3 ASCII characters/string '000' to '359'
N2ASCI:	ldi	TMP1,'0'		; (r8,r9) is now nr between 0 and 359 deg
		mov r1,TMP1			; convertion to ASCII string from binary number
		mov r2,TMP1
		mov	r3,TMP1
		mov	TMP1,r8
		mov	TMP2,r9
A2D6:	subi	TMP1,100
		sbci	TMP2,0
		brcs	A2D7
		inc r1				; hunderts
		mov	r8,TMP1
		mov	r9,TMP2
		rjmp	A2D6
A2D7:	mov	TMP1,r8
A2D9:	subi	TMP1,10
		brcs	A2D8
		inc	r2				; tenths
		mov r8,TMP1
		rjmp	A2D9
A2D8:	add	r3,r8			; units
		ret					; 123 -> r1='1' r2='2' r3='3'


AZCONV:
	; convertion from ADC to degrees for Azimuth
		lds	r3,AZAD2
		lds	TMP1,AZAD1
		sub	r3,TMP1
		lds	r4,AZAD2+1
		lds	TMP1,AZAD1+1
		sbc	r4,TMP1
		brcc	AZCO1	; no carry -> [AZAD2]-[AZAD1] positive
		neg	r3
		brcs	AZCO2
		dec	r4
AZCO2:	com	r4			; r3,r4 = - r3,r4 = [AZAD1]-[AZAD2]
		lds	r1,AZAD1
		cli				;consistency protect
		lds	TMP1,AZADC
		sub r1,TMP1
		lds	r2,AZAD1+1
		lds	TMP1,AZADC+1
		sei
		sbc	r2,TMP1		; r1,r2 = [AZAD1]-[AZADC]
		rjmp	AZCO3
AZCO1:	cli
		lds	r1,AZADC
		lds	TMP1,AZAD1
		sub r1,TMP1
		lds	r2,AZADC+1
		sei
		lds	TMP1,AZAD1+1
		sbc	r2,TMP1		; r1,r2 = [AZADC]-[AZAD1]
AZCO3:	brcc	AZCO4
		clr	r8			; when r1,r2 negative (bellow CCW switch) - should not happen, but...
		clr	r9			; output taken as CCW switch
		rjmp	AZCO5
AZCO4:	lds	r5,AZMAX
		lds	r6,AZMAX+1
		rcall	XCALC	;(r1,r2)/(r3,r4)*(r5,r6)
AZCO5:	lds	TMP1,AZSW1
		add	r8,TMP1
		lds	TMP1,AZSW1+1
		adc	r9,TMP1		;r8,r9 += [AZSW1]
		sts	AZDEG,r8
		sts	AZDEG+1,r9	;store result
		ret
		
ELCONV:
	; new version of convertion from ADC to degrees for Elevation
		lds	r3,ELAD2
		lds	TMP1,ELAD1
		sub	r3,TMP1
		lds	r4,ELAD2+1
		lds	TMP1,ELAD1+1
		sbc	r4,TMP1
		brcc	ELCO1	; no carry -> [ELAD2]-[ELAD1] positive
		neg	r3
		brcs	ELCO2
		dec	r4
ELCO2:	com	r4			; r3,r4 = - r3,r4 = [ELAD1]-[ELAD2]
		lds	r1,ELAD1
		cli				;data consistency protect (ELADC)
		lds	TMP1,ELADC
		sub r1,TMP1
		lds	r2,ELAD1+1
		lds	TMP1,ELADC+1
		sei
		sbc	r2,TMP1		; r1,r2 = [ELAD1]-[ELADC]
		rjmp	ELCO3
ELCO1:	cli	
		lds	r1,ELADC
		lds	TMP1,ELAD1
		sub r1,TMP1
		lds	r2,ELADC+1
		sei
		lds	TMP1,ELAD1+1
		sbc	r2,TMP1		; r1,r2 = [ELADC]-[ELAD1]
ELCO3:	brcc	ELCO4
		clr	r8			; when r1,r2 negative (bellow CCW switch) - should never happen, but...
		clr	r9			; output taken as CCW switch
		rjmp	ELCO5
ELCO4:	lds	r5,ELMAX
		clr	r6
		rcall	XCALC	;(r1,r2)/(r3,r4)*(r5,r6)
ELCO5:	lds	TMP1,ELSW1
		add	r8,TMP1
		clr	TMP1
		adc	r9,TMP1		;r8,r9 += [ELSW1]
		breq	ELCO6	;r9 should be always zero
		lds	r8,ELSW1	;but if not, UP end switch value is taken ([ELSW1]+[ELMAX])
		lds	TMP1,ELMAX
		add	r8,TMP1
ELCO6:	sts	ELDEG,r8	;store result
		ret
		
	
; 16 bit unsigned integer calculation: (r1,r2)/(r3,r4)*(r5,r6)
; limitation: r1,2<r3,r4; r5,r6<=1024
; result = r8,r9
XCALC:		
		clr	r7				;clear result buffer
		clr	r8
		clr	r9
		ldi	TMP2,13			; TMP2 - loop counter
XC1:	lsl	r7				;result*=2
		rol	r8
		rol	r9
		mov	r10,r1			; (r10,r11) = (r1,r2) - (r3,r4)
		sub	r10,r3
		mov	r11,r2
		sbc	r11,r4			; (r1,r2) >= (r3,r4) ... no carry
		brcs	XC2
		mov	r1,r10			; (r1,r2) -= (r3,r4)
		mov	r2,r11
		add	r7,r5
		adc	r8,r6
		clr	TMP1
		adc	r9,TMP1
XC2:	lsl	r1				; (r1,r2) *= 2
		rol	r2
		dec	TMP2
		brne	XC1
		ldi	TMP2,4			; result is now degress shifted by 12 (4+8) bits left
XC3:	lsr	r9
		ror	r8
		dec	TMP2
		brne	XC3
		brcc	XC4			;done; just round the result...
		ldi	TMP1,1
		add	r8,TMP1
		clr	TMP1
		adc	r9,TMP1
XC4:	ret		
		
		
		
;******* PRINT NUMBER ********
; TMP1 - position of text (command)
; r1,r2,r3 - 3 ASCII characters to be printed
PRNR:	rcall	LCDC
		mov	TMP1,r1
		rcall	LCDD
		mov	TMP1,r2
		rcall	LCDD
		mov	TMP1,r3
		rcall	LCDD
		ret
		
;****** DECODE NUMBER RECEIVED FROM COMPUTER (0..999) ********
;input: pointer to RX buffer to first numeric char in r30
;       end defined by first non numeric char or end of rcvd tlg
;		when the number begins with minus sign, result is ZERO (can happen with negative EL)
;output: ASCII characters in r1(hunderts),r2,r3; numeric value in r4(LSB),r5
;		r30 - first non-numeric char or RXBP
PARNR:	ldi	TMP2,'0'
		mov	r1,TMP2		;set to '000'
		mov	r2,TMP2
		mov	r3,TMP2
		lds	TMP3,RXBP	;expected end of text
		clr	r31			; Z= pointer to buffer
		ld	TMP1,Z
		clr	r4
		cpi	TMP1,'-'
		brne	PAR1
		inc	r30
		inc	r4			;r4 = 1 means number begins with minus sign -> result is 0
PAR1:	cp	r30,TMP3
		brcc	PAR2	;we are at end (r30 >= RXBP)
		ld	TMP1,Z		;char into TMP1
		cpi	TMP1,'0'
		brcs	PAR2	;jump when <'0'
		cpi	TMP1,'9'+1
		brcc PAR2		;jump when >'9'
		and	r4,r4
		brne PAR3
		mov	r1,r2
		mov r2,r3
		mov r3,TMP1		;shift the string
PAR3:	inc	r30
		rjmp	PAR1
PAR2:	clr	r5			;conversion from ASCII string to number
		mov	TMP1,r1
		subi	TMP1,'0'	;hunderth h
		mov r4,TMP1
		lsl	r4
		lsl	r4			;r4 *= 4
		add	r4,TMP1		;r4 += h (r5=5*h)
		lsl	r4			;r4 *= 2  (r5=10*h)  0..90
		mov	TMP1,r2
		subi	TMP1,'0'; tenths t
		add	r4,TMP1		; 0..99  (10*h+t)
		mov	TMP1,r4
		lsl	r4			;r5*=2 0..198
		lsl	r4			;r5*=2 (40*h+4*t) 0..396 (256+140)  carry!!!
		rol	r5			;carry handling
		add	r4,TMP1
		clr	TMP1
		adc	r5,TMP1		;carry from addition  r4,5= (50*h+5*t)
		lsl	r4
		rol	r5			;r4,5= 100*h+10*t
		mov	TMP1,r3		;units u
		subi	TMP1,'0'
		add	r4,TMP1
		clr	TMP1
		adc	r5,TMP1		;r4,5= 100*h+10*t+u
		ret

.equ	CCW1	=0x80	;bit 7 - CCW direction (SP > current AZ)
.equ	NEAR1	=0x40	;bit 6 - SP - current AZ <= 180 deg
.equ	RNG1	=0x20	;bit 5 - SP in rotation range
.equ	CCW2	=0x08	;bit 3 - former CCW direction (SP > current AZ)
.equ	NEAR2	=0x04	;bit 2 - former SP - current AZ <= 180 deg
.equ	RNG2	=0x02	;bit 1 - former SP in rotation range

;***** TARGET AZIMUTH CALCULATION ******
; criteria: 1. must be in rotation range
;           2. closest to current position
; input: rotator parameters, current AZ, command AZ
; output: AZ setpoint (AZSP) in degrees

AZC:	lds	r11,AZCMD		; (r11,r12)=[AZCMD] - 360  = sure more CCW from current AZ
		ldi	TMP1,104
		sub	r11,TMP1
		lds	r12,AZCMD+1
		ldi	TMP1,1
		sbc	r12,TMP1
		clr	ATTR
AZC1:	lds	r5,AZDEG		; current AZ
		lds	r6,AZDEG+1
		sub	r5,r11
		sbc	r6,r12			; current AZ - possible SP
		brpl	AZC2		; when positive
		sbr	ATTR,CCW1
		neg	r5				; (r5,r6) = - (r5,r6)
		brcs	AZC8
		dec	r6
AZC8:	com	r6
AZC2:	and r6,r6
		brne	AZC3
		ldi TMP1,180
		cp	TMP1,r5
		brcs	AZC3		; CY when > 180
		sbr	ATTR,NEAR1
AZC3:	sbrc	ATTR,7		; CCW ?
		rjmp	AZC4
		mov	r13,r11			; store old possible SP to (r13,r14)
		mov	r14,r12
		swap	ATTR
		ldi	TMP1,104		; next SP value = old + 360 deg
		add	r11,TMP1
		ldi	TMP1,1
		adc	r12,TMP1
		rjmp	AZC1
AZC4:	lds	r5,AZSW1
		mov	r7,r5
		lds	r6,AZSW1+1
		mov	r8,r6
		sub	r7,r13
		sbc	r8,r14			;  (r7,r8) = CCW switch - older SP
		brmi	AZC10
		mov	TMP1,r7
		or	TMP1,r8
		brne	AZC5		; zero difference is also in range
AZC10:	sbr	ATTR,RNG2		; older SP in rotator range
AZC5:	lds	TMP1,AZMAX
		add	TMP1,r5
		lds	TMP2,AZMAX+1
		adc	TMP2,r6			; (TMP) ... CW switch AZ
		mov	r5,r11
		mov	r6,r12
		sub	r5,TMP1
		sbc	r6,TMP2			; (r5,r6) = newer SP - CW switch 
		brmi	AZC9
		mov	TMP1,r5
		or	TMP1,r6
		brne	AZC6		; zero diff is also in
AZC9:	sbr	ATTR,RNG1		; new SP in rotator range
AZC6:	sbrs	ATTR,6		; NEAR1 decision: we have 2 possible SP on each side of current AZ
		rjmp	AZC7
		sbrc	ATTR,5		; new closer; RNG1?
		rjmp	AZCNEW		; YES NEW!!!!
		sbrc	ATTR,1		; RNG2
		rjmp	AZCOLD		; YES OLD!!!
		rjmp	AZCXX		; both new and old out of range
AZC7:	sbrc	ATTR,1		; old closer; RNG2?
		rjmp	AZCOLD		; YES OLD!!!
		sbrc	ATTR,5		; RNG1
		rjmp	AZCNEW		; YES NEW!!
AZCXX:	cp	r5,r7
		cpc	r6,r8			; CW zone - CCW zone
		lds	r11,AZSW1
		lds	r12,AZSW1+1		; CCW value ready
		brpl	AZC12		; turn CCW!!!
		lds	TMP1,AZMAX
		add	r11,TMP1
		lds	TMP1,AZMAX+1
		adc	r12,TMP1			; CW value ready
		rjmp	AZCNEW
AZCOLD:	mov	r11,r13
		mov	r12,r14
AZC12:	cbr	RFLAG,1<<2	; flag cleared -> CCW direction
		rjmp	AZC11
AZCNEW:	sbr	RFLAG,1<<2	; flag set -> CW direction
AZC11:	sts	AZSP,r11			; store result
		sts	AZSP+1,r12
		ret	

;AZCSIM:		;simulation version - just simple copy [AZCMD] to [AZSP]
;		lds	TMP1,AZCMD
;		sts	AZSP,TMP1
;		lds	TMP2,AZCMD+1
;		sts	AZSP+1,TMP2
;		lds TMP3,AZDEG+1
;		sub	TMP2,TMP3	; compare MSB
;		brne	AZCSIM1
;		lds	TMP3,AZDEG
;		sub	TMP1,TMP3	; CY cleared when [AZSP]>=[AZDEG] (CW direction)
;AZCSIM1:cbr	RFLAG,1<<2	; flag cleared -> CCW direction
;		brcs	AZCSIM2
;		sbr	RFLAG,1<<2	; flag set -> CW direction
;AZCSIM2:ret


; ***** ELEVATION SETPOINT CALCULATION *********
; no inp/out - all data taken/stored in SRAM
ELC:	; calculates setpoint, direction
		lds	TMP1,ELCMD
		lds	TMP2,ELSW1
		cp	TMP1,TMP2
		brcc	ELC1
		mov	TMP1,TMP2	; command less than DWN end limit - set to DWN end limit
		rjmp	ELC2
ELC1:	lds	TMP3,ELMAX
		add	TMP2,TMP3	; deg value of UP end limit
		cp	TMP2,TMP1
		brcc	ELC2
		mov	TMP1,TMP2	; command more than UP limit - set to up limit
ELC2:	sts	ELSP,TMP1	; store corrected value
		lds	TMP2,ELDEG
		cp	TMP1,TMP2	;CY cleared when [ELSP]>=[ELDEG] (UP direction)
		cbr	RFLAG,1<<4	; flag cleared -> DWN direction
		brcs	ELC3
		sbr	RFLAG,1<<4	; flag set -> UP direction
ELC3:	ret
		
		
;ELCSIM:		;simulation version only - copies ELCMD to ELSP and sets direction flag !!!
;		lds	TMP1,ELDEG
;		lds	TMP2,ELCMD
;		sts	ELSP,TMP2
;		sub	TMP2,TMP1	;CY cleared when [ELSP]>=[ELDEG] (UP direction)
;		cbr	RFLAG,1<<4	; flag cleared -> DWN direction
;		brcs	ELCSIM1
;		sbr	RFLAG,1<<4	; flag set -> UP direction
;ELCSIM1:ret
		
		
;*******  RESTORE SETUP DATA FROM EEPROM TO SRAM *******	
EERD:	clr	r31
		ldi	r30,ESTART	; starting address in SRAM to Z
		ldi TMP2,CHKSEED	; checksum register
		clr	TMP1		; EEPROM address (begin of EEPROM)
EERD1:	out	EEAR,TMP1
		sbi	EECR,EERE	; command start read
EERD2:	sbic	EECR,EERE
		rjmp	EERD2
		in	TMP3,EEDR	; data available
		eor	TMP2,TMP3	; checksum calculation
		inc	TMP1
		cpi	TMP1,EELEN+1	; test end of eeprom data
		brcc	EERD3
		st	Z+,TMP3		; data -> SRAM
		rjmp	EERD1
EERD3:	and	TMP2,TMP2	; zero = EEPROM data valid
		ret
		
;******   SAVE SETUP DATA FROM SRAM TO EEPROM ********
EEWR:	clr	r31
		ldi	r30,ESTART
		ldi	TMP2,CHKSEED
		clr	TMP1
EEWR1:	sbic	EECR,EEWE
		rjmp	EEWR1
		out	EEAR,TMP1
		ld	TMP3,Z+
		inc	TMP1
		cpi	TMP1,EELEN+1
		brcc	EEWR3
		eor	TMP2,TMP3		
		out	EEDR,TMP3
		sbi	EECR,EEMWE
		sbi	EECR,EEWE
		rjmp	EEWR1
EEWR3:	out	EEDR,TMP2
		sbi	EECR,EEMWE
		sbi	EECR,EEWE
EEWR2:	sbic	EECR,EEWE
		rjmp	EEWR2
		ret

WKR:	; WAIT for KEY release - returns when no key is pressed, otherwise in loop
		lds	TMP1,KEYID
		andi	TMP1,3	; just 2 lower bits
		brne	WKR		; LOOP
		sts	KEYID,TMP1	; clear timeout flags
		ret
		
ENTNR:		;******* ENTER NUMBER USING KEYBOARD ********
; number selected using UP / DWN keys
; input r4,r5 - initial value
;		r6,7 - upper limit (posible values 0 to upper limit)
; output r4,r5 - selected value
		mov	r8,r4
		mov	r9,r5
		rcall	N2ASCI
		ldi	TMP1,LCDSET
		rcall	PRNR		
EN3:	cli
		lds	TMP1,KEYID
		mov	TMP2,TMP1
		andi	TMP1,0x03
		sts	KEYID,TMP1		;clear b2,b3 in keyid
		sei
		andi	TMP2,0x0C
		breq	EN3			;jump when both 0 - loop
		ldi	TMP3,1			;step = 1
		sbrc	TMP2,3
		ldi	TMP3,10			;step = 10
		
		cpi	TMP1,3			; test Switch 3 (DWN)
		brne	EN1
		clr	TMP1
		sub	r4,TMP3
		sbc	r5,TMP1			; - step
		brcc	ENTNR		;after decrement still no CY
		clr	r4
		clr	r5				;lowest value = 0
		rjmp	ENTNR		; print the number
		
EN1:	cpi	TMP1,2			;test Switch 2 (UP)
		brne	EN2
		clr	TMP1
		add	r4,TMP3
		adc	r5,TMP1			; + step
		cp	r7,r5
		brne	EN4
		cp	r6,r4
EN4:	brcc	ENTNR		;no CY when r6,r7 >= r4,r5
		mov	r4,r6
		mov	r5,r7			;r4,r5 = upper limit
		rjmp	ENTNR
		
EN2:	cpi	TMP1,1			;test Switch 1 (finish)
		brne	EN3
		ret					

ADCSET:	;****  PRINT ADC VALUE - used during setup
		; input: r30 - address of registers containing the value in SRAM
		sbrs	RFLAG,5	; new ADC value ready
		rjmp	ADCS1
		cbr	RFLAG,1<<5
		clr	r31
		cli
		ld	r8,Z+
		ld	r9,Z
		sei
		dec	r30			; back to original value
		rcall	N2ASCI
		ldi	TMP1,LCDADC
		rcall	PRNR
ADCS1:	lds	TMP1,KEYID
		cpi	TMP1,5		; S1 key pressed for 100 msec
		brne	ADCSET
		ldi	TMP1,TXT15	;clear the value
		rjmp	TEXT	; call and return

		
SETUP:	;******* ROTOR PARAMETER SETUP
		ldi	TMP1,_PORTC
		out	PORTC,TMP1	;stop motor
		ldi	TMP1,1
		rcall	LCDC	;clear LCD
		ldi	TMP1,50
		rcall	WAIT	;wait 5 msec

		ldi	TMP1,TXT16
		rcall	TEXT	; "Dead zone:"		
		rcall	WKR
		lds	r4,DEAD
		clr r5			; initial value = DEAD
		ldi	TMP1,30
		mov	r6,TMP1
		clr	r7			; upper limit = r6/r7 = 30 deg
		rcall	ENTNR
		sts	_DEAD,r4		
	
		ldi	TMP1,TXT3
		rcall	TEXT	; "AZIMUTH SETUP"
		ldi	TMP1,TXT5
		rcall TEXT		; "TURN"
		ldi	TMP1,TXT6
		rcall TEXT		; "CCW"
		ldi	TMP1,TXT10
		rcall	TEXT	; "LIMIT"
		rcall	WKR
		ldi	r30,AZADC
		rcall	ADCSET	; waiting loop displaying current ADC AZ value
		lds	TMP1,AZADC
		sts	_AZAD1,TMP1
		lds TMP1,AZADC+1
		sts	_AZAD1+1,TMP1

		ldi	TMP1,TXT11
		rcall	TEXT	; "Degrees:"		
		rcall	WKR
		lds	r4,AZSW1
		lds	r5,AZSW1+1			; initial value = 0
		ldi	TMP1,0x67
		mov	r6,TMP1
		ldi	TMP1,1
		mov	r7,TMP1		; upper limit = r6/r7 = 359
		rcall	ENTNR
		sts	_AZSW1,r4
		sts	_AZSW1+1,r5

		ldi	TMP1,TXT5
		rcall	TEXT	; "TURN"
		ldi	TMP1,TXT7
		rcall	TEXT	; "CW"
		ldi	TMP1,TXT10
		rcall	TEXT	; "LIMIT"
		rcall	WKR
		ldi	r30,AZADC
		rcall	ADCSET	; waiting loop displaying current ADC AZ value
		lds	TMP1,AZADC
		sts	_AZAD2,TMP1
		lds TMP1,AZADC+1
		sts	_AZAD2+1,TMP1

		ldi	TMP1,TXT12
		rcall	TEXT	; "Rot.range:"
		rcall	WKR
		lds	r4,AZMAX
		lds r5,AZMAX+1			; initial value
		ldi	TMP1,0xe7
		mov	r6,TMP1
		ldi	TMP1,3
		mov	r7,TMP1		; upper limit = r6/r7 = 999
		rcall	ENTNR
		sts	_AZMAX,r4
		sts	_AZMAX+1,r5

		ldi	TMP1,TXT4
		rcall	TEXT	; "ELEVATION"
		ldi	TMP1,TXT5
		rcall TEXT		; "TURN"
		ldi	TMP1,TXT8
		rcall TEXT		; "DWN"
		ldi	TMP1,TXT10
		rcall	TEXT	; "LIMIT"
		rcall	WKR
		ldi	r30,ELADC
		rcall	ADCSET	; waiting loop displaying current ADC EL value
		lds	TMP1,ELADC
		sts	_ELAD1,TMP1
		lds TMP1,ELADC+1
		sts	_ELAD1+1,TMP1		
		
		ldi	TMP1,TXT11
		rcall	TEXT	; "Degrees:"		
		rcall	WKR
		lds	r4,ELSW1
		clr	r5			; initial value
		ldi	TMP1,60
		mov	r6,TMP1
		clr r7			; upper limit = r6/r7 = 60 deg
		rcall	ENTNR
		sts	_ELSW1,r4

		ldi	TMP1,TXT5
		rcall	TEXT	; "TURN"
		ldi	TMP1,TXT9
		rcall	TEXT	; "UP"
		ldi	TMP1,TXT10
		rcall	TEXT	; "LIMIT"
		rcall	WKR
		ldi	r30,ELADC
		rcall	ADCSET	; waiting loop displaying current ADC EL value
		lds	TMP1,ELADC
		sts	_ELAD2,TMP1
		lds TMP1,ELADC+1
		sts	_ELAD2+1,TMP1

		ldi	TMP1,TXT12
		rcall	TEXT	; "Rot.range:"
		rcall	WKR
		lds	r4,ELMAX
		clr r5			; initial value
		ldi	TMP1,180
		mov	r6,TMP1
		clr	r7			; upper limit = r6/r7 = 180
		rcall	ENTNR
		sts	_ELMAX,r4

		ldi	TMP1,TXT13
		rcall	TEXT	;"SETUP DATA STORE"
		ldi	TMP1,TXT14
		rcall	TEXT	;" CANCEL"
		rcall	WKR
SE1:	cli
		lds	TMP1,KEYID
		ldi	TMP2,3
		and	TMP2,TMP1
		sts	KEYID,TMP2
		sei
		cpi	TMP2,2		; S2 switch for more than 2 sec (STORE)
		brne	SE2
		sbrs	TMP1,3
		rjmp	SE2
		clr	r31
		ldi	r30,_AZSW1	; source
		ldi	TMP3,EELEN	; loop counter
		cli				; disable interrupt because of possible Y register collision
		clr	r29
		ldi	r28,AZSW1	; destination
SE4:	ld	TMP1,Z+		; copying temporary cnfg data over real data
		st	Y+,TMP1
		dec	TMP3
		brne	SE4
		sei
		rcall	EEWR	; store the WHOLE setup data into EEPROM
		rjmp	INITXT
SE2:	cpi	TMP2,3		; S3 switch for more than 100 msec (CANCEL)
		brne	SE1
		sbrs	TMP1,2
		rjmp	SE1
SE3:	rjmp	INITXT	;print initial LCD text and return

		
.ESEG
		
.ORG	0x0
;**** DEFAULT ROTOR SETUP PARAMETERS IN EEPROM *****

		.DB	0x64,0	; AZ end switch CCW degrees 100
		.DB 0xD0,2	; AZ rotation range 720
		.DB 0x32,0	; AZ ADC value for CCW switch 50
		.DB	0xF4,1	; AZ ADC value for CW switch 500
		.DB	5		; EL minimum elevation 5
		.DB 0x64		; EL maximum elevation 100
		.DB	0x14,0	; EL ADC value for min elevation 20
		.DB 0x2C,1	; EL ADC value for max elevation 300
		.DB 5		; dead zone of rotor (if command value is very close to current, no move) 5
		.DB 0x86	; checksum	(XOR of all data + 0xAA)
		
		
;text definitions
; 1st byte defines position on display (it is sent as a command)
; value = 0x80 + 0x40 * line number (0 or 1) + position (0 to 0x0F)
; example: 2nd line, 3rd char position : 0x80 + 0x40 + 0x02 = 0xC2
; ASCII characters follow, terminated by 0

TXT1:	.DB		0x80,'A','z',' ',' ',' ',' ',' ','(','*','*','*',')',' ','S','T','P',0
TXT2:	.DB		0xC0,'E','l',' ',' ',' ',' ',' ','(','*','*','*',')',' ','S','T','P',0
TXT3:	.DB		0x80,'A','Z','I','M','U','T','H',0
TXT4:	.DB		0x80,'E','L','E','V','A','T','I','O','N',0
TXT5:	.DB		0xC0,'T','u','r','n',' ',0
TXT6:	.DB		0xC5,'C','C','W',' ',0
TXT7:	.DB		0xC5,'C','W',' ',' ',0
TXT8:	.DB		0xC5,'D','W','N',' ',0
TXT9:	.DB		0xC5,'U','P',' ',' ',0
TXT10:	.DB		0xC9,'L','i','m','i','t',0
TXT11:	.DB		0xC0,'D','e','g','r','e','e','s',':',' ',' ',' ',' ',' ',' ',0
TXT12:	.DB		0xC0,'R','o','t','.',' ','R','a','n','g','e',':',' ',' ',' ',0
TXT13:	.DB		0x80,'S','e','t','u','p',' ','d','a','t','a',' ','S','T','O','R','E',0
TXT14:	.DB		0xC0,' ',' ',' ',' ',' ',' ',' ',' ',' ',' ','C','A','N','C','E','L',0
TXT15:	.DB		LCDADC,' ',' ',' ',0	;clear LCD where was ADC value
TXT16:	.DB		0xC0,'D','e','a','d',' ','z','o','n','e',':',0
T1STP:	.DB		0x8D,'S','T','P',0
T_CW:	.DB		0x8D,'C','W',' ',0
T_CCW:	.DB		0x8D,'C','C','W',0
T2STP:	.DB		0xCD,'S','T','P',0
T_UP:	.DB		0xCD,'U','P',' ',0
T_DWN:	.DB		0xCD,'D','W','N',0
VERS:	.DB		0x80,'A','V','R','O','T',' ','1','.','2','0',' ','O','K','1','D','X',0
VERT:	.DB		0xC2,'A','N','D',' ','K','E','6','L','V','K',0
